/*
 * Copyright (c) 2022 Apple Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#import <XCTest/XCTest.h>
#include <CoreUtils/CoreUtils.h>

#include "ResourceRecordBytes.h"
#include "domain_name_labels.h"
#include "dns_obj_rr.h"
#include "dns_obj_rr_ds.h"
#include "dns_obj_rr_dnskey.h"
#include "dns_common.h"
#include "dns_obj_crypto.h"
#include <string.h>
#include <stdint.h>

#include "mdns_strict.h"

@interface DNSObjRRTest : XCTestCase

@end

@implementation DNSObjRRTest

//======================================================================================================================
// MARK: - Test Data and Expected Result

const resource_record_bytes_t test_cases_rr[NUM_OF_RR_RECORDS] = {
	//==================================================================================================================
	// MARK: - CNAME
	{
		// This is a pseudo CNAME record that is used to mark a start of a CNAME chain.
		// For example, this pseudo CNAME record marks the start of the CNAME chain below:
		// www.apple.com.										1796	IN CNAME www.apple.com.edgekey.net.
		// www.apple.com.edgekey.net.							21596	IN CNAME www.apple.com.edgekey.net.globalredir.akadns.net.
		// www.apple.com.edgekey.net.globalredir.akadns.net.	3596	IN CNAME e6858.dscx.akamaiedge.net.
		.name = {0},
		.type = kDNSRecordType_CNAME,
		.class = kDNSClassType_IN,
		.rdata_len = 1,
		.rdata = {0},
		.expected_result_u.cname = {0},
	},
	{
		.name = {
			3, 'w', 'w', 'w', 5, 'a', 'p', 'p', 'l', 'e', 3, 'c', 'o', 'm', 0
		},
		.type = kDNSRecordType_CNAME,
		.class = kDNSClassType_IN,
		.rdata_len = 27,
		.rdata = {
			0x03, 0x77, 0x77, 0x77, 0x05, 0x61, 0x70, 0x70, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x07, 0x65, 0x64, 0x67,
			0x65, 0x6b, 0x65, 0x79, 0x03, 0x6e, 0x65, 0x74, 0x00
		},
		.expected_result_u.cname = {
			3, 'w', 'w', 'w', 5, 'a', 'p', 'p', 'l', 'e', 3, 'c', 'o', 'm', 7, 'e', 'd', 'g', 'e', 'k', 'e', 'y',
			3, 'n', 'e', 't', 0
		},
	},
	{
		.name = {
			3, 'w', 'w', 'w', 5, 'a', 'p', 'p', 'l', 'e', 3, 'c', 'o', 'm', 7, 'e', 'd', 'g', 'e', 'k', 'e', 'y',
			3, 'n', 'e', 't', 0
		},
		.type = kDNSRecordType_CNAME,
		.class = kDNSClassType_IN,
		.rdata_len = 50,
		.rdata = {
			0x03, 0x77, 0x77, 0x77, 0x05, 0x61, 0x70, 0x70, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x07, 0x65, 0x64, 0x67,
			0x65, 0x6b, 0x65, 0x79, 0x03, 0x6e, 0x65, 0x74, 0x0b, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x72, 0x65, 0x64,
			0x69, 0x72, 0x06, 0x61, 0x6b, 0x61, 0x64, 0x6e, 0x73, 0x03, 0x6e, 0x65, 0x74, 0x00
		},
		.expected_result_u.cname = {
			3, 'w', 'w', 'w', 5, 'a', 'p', 'p', 'l', 'e', 3, 'c', 'o', 'm', 7, 'e', 'd', 'g', 'e', 'k', 'e', 'y',
			3, 'n', 'e', 't', 11, 'g', 'l', 'o', 'b', 'a', 'l', 'r', 'e', 'd', 'i', 'r',
			6, 'a', 'k', 'a', 'd', 'n', 's', 3, 'n', 'e', 't', 0
		},
	},
	{
		.name = {
			3, 'w', 'w', 'w', 5, 'a', 'p', 'p', 'l', 'e', 3, 'c', 'o', 'm', 7, 'e', 'd', 'g', 'e', 'k', 'e', 'y',
			3, 'n', 'e', 't', 11, 'g', 'l', 'o', 'b', 'a', 'l', 'r', 'e', 'd', 'i', 'r',
			6, 'a', 'k', 'a', 'd', 'n', 's', 3, 'n', 'e', 't', 0
		},
		.type = kDNSRecordType_CNAME,
		.class = kDNSClassType_IN,
		.rdata_len = 27,
		.rdata = {
			0x05, 0x65, 0x36, 0x38, 0x35, 0x38, 0x04, 0x64, 0x73, 0x63, 0x78, 0x0a, 0x61, 0x6b, 0x61, 0x6d, 0x61, 0x69,
			0x65, 0x64, 0x67, 0x65, 0x03, 0x6e, 0x65, 0x74, 0x00
		},
		.expected_result_u.cname = {
			5, 'e', '6', '8', '5', '8', 4, 'd', 's', 'c', 'x', 10, 'a', 'k', 'a', 'm', 'a', 'i', 'e', 'd', 'g', 'e',
			3, 'n', 'e', 't', 0
		},
	},
	//==================================================================================================================
	// MARK: - SOA
	{
		.name = {
			3, 'c', 'd', 'c', 3, 'g', 'o', 'v', 0
		},
		.type = kDNSRecordType_SOA,
		.class = kDNSClassType_IN,
		.rdata_len = 53,
		.rdata = {
			0x03, 0x6e, 0x73, 0x32, 0x03, 0x63, 0x64, 0x63, 0x03, 0x67, 0x6f, 0x76, 0x00, 0x0a, 0x68, 0x6f, 0x73, 0x74,
			0x6d, 0x61, 0x73, 0x74, 0x65, 0x72, 0x03, 0x63, 0x64, 0x63, 0x03, 0x67, 0x6f, 0x76, 0x00, 0x26, 0xac, 0x25,
			0xce, 0x00, 0x00, 0x2a, 0x30, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x09, 0x3a, 0x80, 0x00, 0x00, 0x0e, 0x10
		},
		.expected_result_u.soa = {
			.minimum_ttl = 3600
		}
	},
	//==================================================================================================================
	// MARK: - SRV
	{
		.name = {
			4, 'W', 'O', 'W', 'G', 8, '_', 'a', 'i', 'r', 'p', 'l', 'a', 'y', 4, '_', 't', 'c', 'p',
			5, 'l', 'o', 'c', 'a', 'l', 0
		},
		.type = kDNSRecordType_SRV,
		.class = kDNSClassType_IN,
		.rdata_len = 27,
		.rdata = {
			0x00, 0x00, 0x00, 0x00, 0x1b, 0x58, 0x12, 0x53, 0x6f, 0x6e, 0x6f, 0x73, 0x2d, 0x33, 0x38, 0x34, 0x32, 0x30,
			0x42, 0x39, 0x32, 0x31, 0x41, 0x42, 0x32, 0xc0, 0x1f
		},
		.expected_result_u.srv = {
			.priority = 0,
			.weight = 0,
			.port = 7000,
			.target = {
				18, 's', 'o', 'n', 'o', 's', '-', '3', '8', '4', '2', '0', 'b', '9', '2', '1', 'a', 'b', '2',
				5, 'l', 'o', 'c', 'a', 'l', 0
			}
		}
	},
	//==================================================================================================================
	// MARK: - DS
	// DS for qdeng.io.
	{
		.name = {
			5, 'q', 'd', 'e', 'n', 'g', 2, 'i', 'o', 0
		},
		.type = kDNSRecordType_DS,
		.class = kDNSClassType_IN,
		.rdata_len = 24,
		.rdata = {
			0xd9, 0xdf, 0x0d, 0x01, 0xcd, 0x82, 0x71, 0xee, 0x0b, 0x8d, 0x54, 0x5b, 0x83, 0x18, 0xee, 0x32, 0x79, 0x5d,
			0x0c, 0xc6, 0xab, 0xa6, 0xe0, 0x3c
		},
		.expected_result_u.ds = {
			.key_tag = 55775,
			.algorithm = DNSKEY_ALGORITHM_ECDSAP256SHA256,
			.digest_type = DS_DIGEST_SHA_1,
			.digest = {
				0xcd, 0x82, 0x71, 0xee, 0x0b, 0x8d, 0x54, 0x5b, 0x83, 0x18, 0xee, 0x32, 0x79, 0x5d, 0x0c, 0xc6, 0xab,
				0xa6, 0xe0, 0x3c
			},
			.digest_length = 20,
			.is_valid_for_dnssec = true,
		}
	},
	// DS for com.
	{
		.name = {
			3, 'c', 'o', 'm', 0
		},
		.type = kDNSRecordType_DS,
		.class = kDNSClassType_IN,
		.rdata_len = 36,
		.rdata = {
			0x78, 0xbd, 0x08, 0x02, 0xe2, 0xd3, 0xc9, 0x16, 0xf6, 0xde, 0xea, 0xc7, 0x32, 0x94, 0xe8, 0x26, 0x8f, 0xb5,
			0x88, 0x50, 0x44, 0xa8, 0x33, 0xfc, 0x54, 0x59, 0x58, 0x8f, 0x4a, 0x91, 0x84, 0xcf, 0xc4, 0x1a, 0x57, 0x66
		},
		.expected_result_u.ds = {
			.key_tag = 30909,
			.algorithm = DNSKEY_ALGORITHM_RSASHA256,
			.digest_type = DS_DIGEST_SHA_256,
			.digest = {
				0xe2, 0xd3, 0xc9, 0x16, 0xf6, 0xde, 0xea, 0xc7, 0x32, 0x94, 0xe8, 0x26, 0x8f, 0xb5, 0x88, 0x50, 0x44,
				0xa8, 0x33, 0xfc, 0x54, 0x59, 0x58, 0x8f, 0x4a, 0x91, 0x84, 0xcf, 0xc4, 0x1a, 0x57, 0x66
			},
			.digest_length = 32,
			.is_valid_for_dnssec = true,
		}
	},
	//==================================================================================================================
	// MARK: - NSEC
	{
		.name = {
			4, 'b', 'l', 'o', 'g', 4, 'r', 'o', 'o', 't', 2, 'c', 'z', 0
		},
		.type = kDNSRecordType_NSEC,
		.class = kDNSClassType_IN,
		.rdata_len = 24,
		.rdata = {
			0x01, 0x2a, 0x04, 0x62, 0x6c, 0x6f, 0x67, 0x04, 0x72, 0x6f, 0x6f, 0x74, 0x02, 0x63, 0x7a, 0x00, 0x00, 0x06, 0x04, 0x00, 0x00, 0x00, 0x00, 0x03

		},
		.expected_result_u.nsec = {
			.current_owner_name = {4, 'b', 'l', 'o', 'g', 4, 'r', 'o', 'o', 't', 2, 'c', 'z', 0},
			.next_owner_name = {1, '*', 4, 'b', 'l', 'o', 'g', 4, 'r', 'o', 'o', 't', 2, 'c', 'z', 0},
			.types_covered = {
				kDNSRecordType_CNAME,
				kDNSRecordType_RRSIG,
				kDNSRecordType_NSEC,
				kDNSRecordType_Invalid
			}
		}
	},
	// MARK: - NSEC
	{
		.name = {
			'4', 'i', 'e', 't', 'f', '3', 'o', 'r', 'g', '0'
		},
		.type = kDNSRecordType_NSEC,
		.class = kDNSClassType_IN,
		.rdata_len = 32,
		.rdata = {
			0x06, 0x5f, 0x64, 0x6d, 0x61, 0x72, 0x63, 0x04, 0x69, 0x65, 0x74, 0x66, 0x03, 0x6f, 0x72, 0x67, 0x00, 0x00,
			0x0d, 0x62, 0x01, 0x80, 0x08, 0x00, 0x03, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10
		},
		.expected_result_u.nsec = {
			.current_owner_name = {'4', 'i', 'e', 't', 'f', '3', 'o', 'r', 'g', '0'},
			.next_owner_name = {6, '_', 'd', 'm', 'a', 'r', 'c', 4, 'i', 'e', 't', 'f', 3, 'o', 'r', 'g', 0},
			.types_covered = {
				kDNSRecordType_A,
				kDNSRecordType_NS,
				kDNSRecordType_SOA,
				kDNSRecordType_MX,
				kDNSRecordType_TXT,
				kDNSRecordType_AAAA,
				kDNSRecordType_RRSIG,
				kDNSRecordType_NSEC,
				kDNSRecordType_DNSKEY,
				kDNSRecordType_SPF,
				kDNSRecordType_Invalid
			}
		}
	},
	//==================================================================================================================
	// MARK: - RRSIG
	{
		.name = {
			3, 'w', 'w', 'w', 4, 'a', 'k', 'a', 'm', 3, 'c', 'd', 'c', 4, 'g', 'o', 'v', 0
		},
			.type = kDNSRecordType_RRSIG,
			.class = kDNSClassType_IN,
			.rdata_len = 160,
			.rdata = {
				0x00, 0x01, 0x0a, 0x04, 0x00, 0x00, 0x00, 0x14, 0x61, 0x5a, 0x0b, 0x2b, 0x61, 0x56, 0x08, 0x9b, 0x99, 0xe6,
				0x04, 0x61, 0x6b, 0x61, 0x6d, 0x03, 0x63, 0x64, 0x63, 0x03, 0x67, 0x6f, 0x76, 0x00, 0x3d, 0xac, 0x5f, 0x36,
				0x26, 0x5a, 0x6d, 0xa6, 0x49, 0xe8, 0xac, 0x27, 0x83, 0xf7, 0x49, 0xd7, 0xee, 0xb3, 0x9b, 0xe3, 0xca, 0x2c,
				0xe2, 0x11, 0x77, 0x2b, 0xf1, 0x81, 0xbf, 0x52, 0x9e, 0xb9, 0x5f, 0xab, 0x9d, 0x85, 0x1e, 0x07, 0xf7, 0xeb,
				0xe8, 0xbd, 0xb3, 0xde, 0x14, 0x9b, 0x92, 0x4b, 0xb4, 0xcf, 0x4c, 0xe2, 0xf3, 0xcd, 0x08, 0xcb, 0x4f, 0xe0,
				0xf7, 0x5d, 0x29, 0x56, 0x5d, 0x4d, 0x90, 0x4a, 0x95, 0xad, 0x2e, 0x52, 0xb4, 0x01, 0x15, 0x7c, 0x6f, 0xfa,
				0x73, 0x13, 0x65, 0x36, 0xc9, 0x16, 0x2d, 0xd0, 0xc8, 0xf8, 0x93, 0xcc, 0x88, 0x18, 0xa0, 0xcd, 0x1a, 0x49,
				0x34, 0xe0, 0xfb, 0x97, 0x5d, 0xa2, 0x9e, 0xbf, 0xce, 0xad, 0x57, 0xca, 0x11, 0xfd, 0x64, 0x04, 0x79, 0xfd,
				0x60, 0x0b, 0x15, 0xfa, 0x7b, 0x55, 0x48, 0x30, 0x2d, 0xc2, 0xee, 0xa5, 0xa8, 0x7a, 0x0c, 0x05
			},
			.expected_result_u.rrsig = {
				.type_covered = kDNSRecordType_A,
				.algorithm = DNSKEY_ALGORITHM_RSASHA512,
				.labels = 4,
				.original_ttl = 20,
				.signature_expiration = 1633291051,
				.signature_inception = 1633028251,
				.key_tag = 39398,
				.signer_name = {4, 'a', 'k', 'a', 'm', 3, 'c', 'd', 'c', 3, 'g', 'o', 'v', 0},
				.signature = {
					0x3d, 0xac, 0x5f, 0x36, 0x26, 0x5a, 0x6d, 0xa6, 0x49, 0xe8, 0xac, 0x27, 0x83, 0xf7, 0x49, 0xd7, 0xee,
					0xb3, 0x9b, 0xe3, 0xca, 0x2c, 0xe2, 0x11, 0x77, 0x2b, 0xf1, 0x81, 0xbf, 0x52, 0x9e, 0xb9, 0x5f, 0xab,
					0x9d, 0x85, 0x1e, 0x07, 0xf7, 0xeb, 0xe8, 0xbd, 0xb3, 0xde, 0x14, 0x9b, 0x92, 0x4b, 0xb4, 0xcf, 0x4c,
					0xe2, 0xf3, 0xcd, 0x08, 0xcb, 0x4f, 0xe0, 0xf7, 0x5d, 0x29, 0x56, 0x5d, 0x4d, 0x90, 0x4a, 0x95, 0xad,
					0x2e, 0x52, 0xb4, 0x01, 0x15, 0x7c, 0x6f, 0xfa, 0x73, 0x13, 0x65, 0x36, 0xc9, 0x16, 0x2d, 0xd0, 0xc8,
					0xf8, 0x93, 0xcc, 0x88, 0x18, 0xa0, 0xcd, 0x1a, 0x49, 0x34, 0xe0, 0xfb, 0x97, 0x5d, 0xa2, 0x9e, 0xbf,
					0xce, 0xad, 0x57, 0xca, 0x11, 0xfd, 0x64, 0x04, 0x79, 0xfd, 0x60, 0x0b, 0x15, 0xfa, 0x7b, 0x55, 0x48,
					0x30, 0x2d, 0xc2, 0xee, 0xa5, 0xa8, 0x7a, 0x0c, 0x05
				}
			}
	},
	//==================================================================================================================
	// MARK: - DNSKEY
	// DNSKEYs from qiaoyu.me.
	{
		.name = {6, 'q', 'i', 'a', 'o', 'y', 'u', 2, 'm', 'e', 0},
		.type = kDNSRecordType_DNSKEY,
		.class = kDNSClassType_IN,
		.rdata_len = 68,
		.rdata = {
			0x01, 0x00, 0x03, 0x0d, 0x05, 0xab, 0xce, 0xa5, 0xa0, 0xde, 0xea, 0x4e, 0xc7, 0xa4, 0xa0, 0x1f, 0x20, 0x0c,
			0xab, 0x59, 0x15, 0x0a, 0x34, 0x4a, 0x07, 0x20, 0x7d, 0x2a, 0x8f, 0x73, 0x51, 0x5c, 0x46, 0x39, 0x28, 0x2d,
			0xc9, 0xb8, 0x55, 0xa8, 0xd5, 0x0c, 0x81, 0x82, 0x28, 0x80, 0xe3, 0x35, 0x4d, 0x69, 0x56, 0x54, 0x00, 0xf7,
			0xd0, 0x28, 0xfa, 0x31, 0x84, 0x9d, 0x9e, 0x7b, 0x53, 0xed, 0x11, 0xbc, 0xd4, 0xe8
		},
		.expected_result_u.dnskey = {
			.flags = DNSKEY_FLAG_ZONE_KEY,
			.protocol = DNSKEY_PROTOCOL_DNSSEC,
			.algorithm = DNSKEY_ALGORITHM_ECDSAP256SHA256,
			.public_key = {
				0x05, 0xab, 0xce, 0xa5, 0xa0, 0xde, 0xea, 0x4e, 0xc7, 0xa4, 0xa0, 0x1f, 0x20, 0x0c, 0xab, 0x59, 0x15,
				0x0a, 0x34, 0x4a, 0x07, 0x20, 0x7d, 0x2a, 0x8f, 0x73, 0x51, 0x5c, 0x46, 0x39, 0x28, 0x2d, 0xc9, 0xb8,
				0x55, 0xa8, 0xd5, 0x0c, 0x81, 0x82, 0x28, 0x80, 0xe3, 0x35, 0x4d, 0x69, 0x56, 0x54, 0x00, 0xf7, 0xd0,
				0x28, 0xfa, 0x31, 0x84, 0x9d, 0x9e, 0x7b, 0x53, 0xed, 0x11, 0xbc, 0xd4, 0xe8
			},
			.public_key_size = ECDSAP_PUBLIC_KEY_BYTES,
			.key_tag = 492,
			.is_zone_key = true,
			.is_secure_entry_point = false,
			.has_supported_algorithm = true,
			.is_valid_for_dnssec = true,
			.priority = 5,
		}
	},
	{
		.name = {6, 'q', 'i', 'a', 'o', 'y', 'u', 2, 'm', 'e', 0},
		.type = kDNSRecordType_DNSKEY,
		.class = kDNSClassType_IN,
		.rdata_len = 68,
		.rdata = {
			0x01, 0x01, 0x03, 0x0d, 0x40, 0x80, 0x83, 0x8b, 0xa5, 0x5e, 0x01, 0x63, 0xa3, 0x22, 0x3c, 0xc5, 0x6a, 0xb5,
			0x44, 0xd2, 0x9d, 0xce, 0xd2, 0x4c, 0x96, 0x06, 0x25, 0xa7, 0xbe, 0x4f, 0x99, 0xf7, 0x67, 0x89, 0x10, 0x12,
			0x26, 0x62, 0xe2, 0xfc, 0x30, 0x0b, 0x98, 0x25, 0xa3, 0x9b, 0x8a, 0x35, 0x72, 0xbd, 0x29, 0xd2, 0xe5, 0xc8,
			0xcb, 0xa3, 0xd4, 0x46, 0x88, 0xdc, 0x33, 0x42, 0x44, 0x9d, 0x79, 0xa3, 0x88, 0xaa
		},
		.expected_result_u.dnskey = {
			.flags = DNSKEY_FLAG_ZONE_KEY | DNSKEY_FLAG_SECURITY_ENTRY_POINT,
			.protocol = DNSKEY_PROTOCOL_DNSSEC,
			.algorithm = DNSKEY_ALGORITHM_ECDSAP256SHA256,
			.public_key = {
				0x40, 0x80, 0x83, 0x8b, 0xa5, 0x5e, 0x01, 0x63, 0xa3, 0x22, 0x3c, 0xc5, 0x6a, 0xb5, 0x44, 0xd2, 0x9d,
				0xce, 0xd2, 0x4c, 0x96, 0x06, 0x25, 0xa7, 0xbe, 0x4f, 0x99, 0xf7, 0x67, 0x89, 0x10, 0x12, 0x26, 0x62,
				0xe2, 0xfc, 0x30, 0x0b, 0x98, 0x25, 0xa3, 0x9b, 0x8a, 0x35, 0x72, 0xbd, 0x29, 0xd2, 0xe5, 0xc8, 0xcb,
				0xa3, 0xd4, 0x46, 0x88, 0xdc, 0x33, 0x42, 0x44, 0x9d, 0x79, 0xa3, 0x88, 0xaa
			},
			.public_key_size = ECDSAP_PUBLIC_KEY_BYTES,
			.key_tag = 7845,
			.is_zone_key = true,
			.is_secure_entry_point = true,
			.has_supported_algorithm = true,
			.is_valid_for_dnssec = true,
			.priority = 5,
		}
	},
	// DNSKEYs from qdeng.io.
	{
		.name = {5, 'q', 'd', 'e', 'n', 'g', 2, 'i', 'o', 0},
		.type = kDNSRecordType_DNSKEY,
		.class = kDNSClassType_IN,
		.rdata_len = 68,
		.rdata = {
			0x01, 0x00, 0x03, 0x0d, 0x29, 0x80, 0xf0, 0x4b, 0xa8, 0x49, 0x8a, 0x1c, 0x5c, 0x0a, 0x52, 0x89, 0x23, 0x06,
			0x5f, 0xd2, 0xde, 0xdc, 0xcf, 0xf9, 0x1f, 0x66, 0x60, 0x1a, 0x03, 0x64, 0xc1, 0x9c, 0x99, 0xc8, 0x2f, 0x77,
			0xdf, 0xec, 0xfc, 0x2f, 0x73, 0x45, 0x49, 0x96, 0x1d, 0x8e, 0x31, 0xfb, 0xf7, 0xd1, 0x39, 0x45, 0x5d, 0x93,
			0xf2, 0x93, 0x68, 0x3c, 0xb6, 0x98, 0x15, 0xdd, 0x55, 0xc3, 0x7c, 0xfa, 0x37, 0xb9
		},
		.expected_result_u.dnskey = {
			.flags = DNSKEY_FLAG_ZONE_KEY,
			.protocol = DNSKEY_PROTOCOL_DNSSEC,
			.algorithm = DNSKEY_ALGORITHM_ECDSAP256SHA256,
			.public_key = {
				0x29, 0x80, 0xf0, 0x4b, 0xa8, 0x49, 0x8a, 0x1c, 0x5c, 0x0a, 0x52, 0x89, 0x23, 0x06, 0x5f, 0xd2, 0xde,
				0xdc, 0xcf, 0xf9, 0x1f, 0x66, 0x60, 0x1a, 0x03, 0x64, 0xc1, 0x9c, 0x99, 0xc8, 0x2f, 0x77, 0xdf, 0xec,
				0xfc, 0x2f, 0x73, 0x45, 0x49, 0x96, 0x1d, 0x8e, 0x31, 0xfb, 0xf7, 0xd1, 0x39, 0x45, 0x5d, 0x93, 0xf2,
				0x93, 0x68, 0x3c, 0xb6, 0x98, 0x15, 0xdd, 0x55, 0xc3, 0x7c, 0xfa, 0x37, 0xb9
			},
			.public_key_size = ECDSAP_PUBLIC_KEY_BYTES,
			.key_tag = 59180,
			.is_zone_key = true,
			.is_secure_entry_point = false,
			.has_supported_algorithm = true,
			.is_valid_for_dnssec = true,
			.priority = 5,
		}
	},
	{
		.name = {5, 'q', 'd', 'e', 'n', 'g', 2, 'i', 'o', 0},
		.type = kDNSRecordType_DNSKEY,
		.class = kDNSClassType_IN,
		.rdata_len = 68,
		.rdata = {
			0x01, 0x01, 0x03, 0x0d, 0x0d, 0xfc, 0x8f, 0x3f, 0x8a, 0xa0, 0xd5, 0x56, 0xb3, 0x2a, 0x3d, 0x44, 0xd3, 0x96,
			0xe5, 0x86, 0xe6, 0xf1, 0xd2, 0x23, 0x3d, 0x17, 0x6d, 0xd6, 0xe8, 0x81, 0xaf, 0xc6, 0x4d, 0x30, 0xb8, 0x32,
			0x01, 0x01, 0x06, 0x5a, 0xa4, 0xb9, 0xcf, 0x59, 0x8f, 0x21, 0x25, 0x9c, 0xb3, 0x8d, 0x61, 0xc6, 0x3c, 0xd1,
			0x7a, 0x2b, 0x77, 0x87, 0xe2, 0xb7, 0x5a, 0x36, 0xe4, 0x8b, 0xe8, 0xca, 0xaf, 0x19
		},
		.expected_result_u.dnskey = {
			.flags = DNSKEY_FLAG_ZONE_KEY | DNSKEY_FLAG_SECURITY_ENTRY_POINT,
			.protocol = DNSKEY_PROTOCOL_DNSSEC,
			.algorithm = DNSKEY_ALGORITHM_ECDSAP256SHA256,
			.public_key = {
				0x0d, 0xfc, 0x8f, 0x3f, 0x8a, 0xa0, 0xd5, 0x56, 0xb3, 0x2a, 0x3d, 0x44, 0xd3, 0x96, 0xe5, 0x86, 0xe6,
				0xf1, 0xd2, 0x23, 0x3d, 0x17, 0x6d, 0xd6, 0xe8, 0x81, 0xaf, 0xc6, 0x4d, 0x30, 0xb8, 0x32, 0x01, 0x01,
				0x06, 0x5a, 0xa4, 0xb9, 0xcf, 0x59, 0x8f, 0x21, 0x25, 0x9c, 0xb3, 0x8d, 0x61, 0xc6, 0x3c, 0xd1, 0x7a,
				0x2b, 0x77, 0x87, 0xe2, 0xb7, 0x5a, 0x36, 0xe4, 0x8b, 0xe8, 0xca, 0xaf, 0x19
			},
			.public_key_size = ECDSAP_PUBLIC_KEY_BYTES,
			.key_tag = 55775,
			.is_zone_key = true,
			.is_secure_entry_point = true,
			.has_supported_algorithm = true,
			.is_valid_for_dnssec = true,
			.priority = 5,
		}
	},
	// DNSKEYs from com.
	{
		.name = {3, 'c', 'o', 'm', 0},
		.type = kDNSRecordType_DNSKEY,
		.class = kDNSClassType_IN,
		.rdata_len = 168,
		.rdata = {
			0x01, 0x00, 0x03, 0x08, 0x03, 0x01, 0x00, 0x01, 0xc5, 0x70, 0xde, 0x9d, 0xcb, 0x87, 0x06, 0xb1, 0x2d, 0xd0,
			0x50, 0x44, 0x0a, 0x17, 0x07, 0x4b, 0xe8, 0xdc, 0x60, 0x37, 0x91, 0x97, 0x32, 0x72, 0x38, 0xc9, 0xf8, 0x3d,
			0xd0, 0x6d, 0x1d, 0x03, 0x99, 0x2c, 0x3e, 0x7f, 0x78, 0x33, 0xb9, 0x39, 0x4f, 0x10, 0x96, 0x14, 0x8a, 0x4d,
			0x9d, 0x33, 0x09, 0xe4, 0x65, 0x23, 0x2a, 0x47, 0x70, 0xcf, 0x9f, 0xbd, 0xdf, 0xec, 0x36, 0x53, 0x54, 0x37,
			0x8b, 0xb5, 0xef, 0x48, 0xad, 0xe6, 0x5f, 0x59, 0xca, 0x78, 0x40, 0xc3, 0x22, 0x9e, 0xb8, 0x05, 0x2d, 0xbb,
			0xcf, 0x6e, 0xb2, 0xbb, 0xac, 0xd2, 0x89, 0x57, 0xa8, 0xfa, 0x8e, 0x98, 0xea, 0x05, 0x0e, 0x73, 0x8b, 0x02,
			0xbd, 0xfb, 0x25, 0x3e, 0x29, 0x7d, 0xad, 0x21, 0x45, 0x72, 0x0e, 0x32, 0x5a, 0x45, 0x1f, 0x8e, 0x4a, 0x23,
			0x89, 0x48, 0xc0, 0x9b, 0x92, 0x1b, 0xc7, 0xfb, 0x9a, 0xb5, 0x12, 0xc1, 0x0f, 0xd0, 0x60, 0x02, 0x28, 0xbe,
			0xf4, 0xaa, 0x7f, 0xe9, 0xb5, 0x17, 0x92, 0xaf, 0xb5, 0xf2, 0x4d, 0x95, 0x39, 0xf1, 0xd7, 0xcc, 0xa6, 0xd3,
			0x18, 0x8f, 0xa9, 0xf4, 0xd4, 0x9f
		},
		.expected_result_u.dnskey = {
			.flags = DNSKEY_FLAG_ZONE_KEY,
			.protocol = DNSKEY_PROTOCOL_DNSSEC,
			.algorithm = DNSKEY_ALGORITHM_RSASHA256,
			.public_key = {
				0x03, 0x01, 0x00, 0x01, 0xc5, 0x70, 0xde, 0x9d, 0xcb, 0x87, 0x06, 0xb1, 0x2d, 0xd0, 0x50, 0x44, 0x0a,
				0x17, 0x07, 0x4b, 0xe8, 0xdc, 0x60, 0x37, 0x91, 0x97, 0x32, 0x72, 0x38, 0xc9, 0xf8, 0x3d, 0xd0, 0x6d,
				0x1d, 0x03, 0x99, 0x2c, 0x3e, 0x7f, 0x78, 0x33, 0xb9, 0x39, 0x4f, 0x10, 0x96, 0x14, 0x8a, 0x4d, 0x9d,
				0x33, 0x09, 0xe4, 0x65, 0x23, 0x2a, 0x47, 0x70, 0xcf, 0x9f, 0xbd, 0xdf, 0xec, 0x36, 0x53, 0x54, 0x37,
				0x8b, 0xb5, 0xef, 0x48, 0xad, 0xe6, 0x5f, 0x59, 0xca, 0x78, 0x40, 0xc3, 0x22, 0x9e, 0xb8, 0x05, 0x2d,
				0xbb, 0xcf, 0x6e, 0xb2, 0xbb, 0xac, 0xd2, 0x89, 0x57, 0xa8, 0xfa, 0x8e, 0x98, 0xea, 0x05, 0x0e, 0x73,
				0x8b, 0x02, 0xbd, 0xfb, 0x25, 0x3e, 0x29, 0x7d, 0xad, 0x21, 0x45, 0x72, 0x0e, 0x32, 0x5a, 0x45, 0x1f,
				0x8e, 0x4a, 0x23, 0x89, 0x48, 0xc0, 0x9b, 0x92, 0x1b, 0xc7, 0xfb, 0x9a, 0xb5, 0x12, 0xc1, 0x0f, 0xd0,
				0x60, 0x02, 0x28, 0xbe, 0xf4, 0xaa, 0x7f, 0xe9, 0xb5, 0x17, 0x92, 0xaf, 0xb5, 0xf2, 0x4d, 0x95, 0x39,
				0xf1, 0xd7, 0xcc, 0xa6, 0xd3, 0x18, 0x8f, 0xa9, 0xf4, 0xd4, 0x9f
			},
			.public_key_size = 164,
			.key_tag = 15549,
			.is_zone_key = true,
			.is_secure_entry_point = false,
			.has_supported_algorithm = true,
			.is_valid_for_dnssec = true,
			.priority = 3,
		}
	},
	{
		.name = {3, 'c', 'o', 'm', 0},
		.type = kDNSRecordType_DNSKEY,
		.class = kDNSClassType_IN,
		.rdata_len = 262,
		.rdata = {
			0x01, 0x01, 0x03, 0x08, 0x01, 0x03, 0xc3, 0xce, 0x57, 0x4d, 0x98, 0xcb, 0xd9, 0x15, 0x7e, 0x0d, 0x70, 0xd2,
			0x74, 0xb8, 0x49, 0xca, 0x0e, 0x0e, 0xed, 0x9a, 0xff, 0xc5, 0xdc, 0xcc, 0x90, 0x47, 0x49, 0x69, 0x06, 0x65,
			0x5c, 0x35, 0xcb, 0x08, 0xb3, 0x3c, 0x4d, 0x17, 0x1b, 0x01, 0x7c, 0xa3, 0x56, 0xf4, 0x96, 0x02, 0x62, 0xaa,
			0x62, 0x93, 0xcd, 0xfa, 0xe8, 0xb1, 0x3b, 0x55, 0xb2, 0x1c, 0x35, 0x1c, 0xdf, 0xa7, 0x68, 0x7d, 0x38, 0xef,
			0x07, 0x46, 0x5f, 0x87, 0xf8, 0x4d, 0x3c, 0xcd, 0xab, 0x8a, 0xf2, 0x4e, 0xde, 0xbd, 0x61, 0x26, 0xbb, 0xfe,
			0xa8, 0x77, 0xed, 0x9b, 0xa2, 0x08, 0x0f, 0xa2, 0x21, 0x1f, 0x18, 0xdc, 0xaf, 0x34, 0xf6, 0x92, 0x23, 0xb1,
			0x4e, 0x22, 0xba, 0x03, 0xb2, 0x7c, 0x3f, 0xb5, 0xa8, 0x20, 0xcc, 0x74, 0x57, 0xd5, 0x9e, 0xd2, 0x3a, 0x23,
			0xa2, 0x3d, 0x63, 0xcd, 0x23, 0x04, 0x94, 0xc9, 0x63, 0x99, 0xef, 0xd5, 0x66, 0x71, 0x0d, 0x46, 0x2e, 0x40,
			0xba, 0x36, 0x56, 0x2f, 0x1b, 0x71, 0xf0, 0x62, 0x6c, 0xa7, 0x42, 0xfe, 0xa8, 0x17, 0x01, 0xaf, 0xfc, 0xa1,
			0x0b, 0x4b, 0x0e, 0xd9, 0x49, 0xda, 0xdb, 0x4d, 0x0d, 0x07, 0x5e, 0xf6, 0x5b, 0xa8, 0xc5, 0x08, 0xec, 0x16,
			0x8c, 0xb2, 0x49, 0xaf, 0x82, 0x6d, 0x46, 0xee, 0x82, 0x99, 0xd5, 0x88, 0x85, 0xec, 0xef, 0x62, 0xa1, 0x53,
			0x5c, 0xd3, 0xee, 0xc0, 0x49, 0xba, 0xa6, 0x64, 0xde, 0xd9, 0xf7, 0xc1, 0x06, 0x53, 0xf4, 0x21, 0xd8, 0xaf,
			0xc1, 0x81, 0x47, 0xbc, 0x1e, 0xcd, 0x17, 0x55, 0xc7, 0x4f, 0x2a, 0xbb, 0x72, 0x62, 0x7a, 0x10, 0x1d, 0xdd,
			0xb2, 0x9c, 0xa3, 0xdc, 0x30, 0xc9, 0x53, 0x12, 0x28, 0x76, 0xff, 0x61, 0xc3, 0x1e, 0x34, 0x4f, 0x27, 0x66,
			0xb2, 0xc0, 0x8a, 0x4a, 0x36, 0x7b, 0xf8, 0xa0, 0xfa, 0x3f
		},
		.expected_result_u.dnskey = {
			.flags = DNSKEY_FLAG_ZONE_KEY | DNSKEY_FLAG_SECURITY_ENTRY_POINT,
			.protocol = DNSKEY_PROTOCOL_DNSSEC,
			.algorithm = DNSKEY_ALGORITHM_RSASHA256,
			.public_key = {
				0x01, 0x03, 0xc3, 0xce, 0x57, 0x4d, 0x98, 0xcb, 0xd9, 0x15, 0x7e, 0x0d, 0x70, 0xd2, 0x74, 0xb8, 0x49,
				0xca, 0x0e, 0x0e, 0xed, 0x9a, 0xff, 0xc5, 0xdc, 0xcc, 0x90, 0x47, 0x49, 0x69, 0x06, 0x65, 0x5c, 0x35,
				0xcb, 0x08, 0xb3, 0x3c, 0x4d, 0x17, 0x1b, 0x01, 0x7c, 0xa3, 0x56, 0xf4, 0x96, 0x02, 0x62, 0xaa, 0x62,
				0x93, 0xcd, 0xfa, 0xe8, 0xb1, 0x3b, 0x55, 0xb2, 0x1c, 0x35, 0x1c, 0xdf, 0xa7, 0x68, 0x7d, 0x38, 0xef,
				0x07, 0x46, 0x5f, 0x87, 0xf8, 0x4d, 0x3c, 0xcd, 0xab, 0x8a, 0xf2, 0x4e, 0xde, 0xbd, 0x61, 0x26, 0xbb,
				0xfe, 0xa8, 0x77, 0xed, 0x9b, 0xa2, 0x08, 0x0f, 0xa2, 0x21, 0x1f, 0x18, 0xdc, 0xaf, 0x34, 0xf6, 0x92,
				0x23, 0xb1, 0x4e, 0x22, 0xba, 0x03, 0xb2, 0x7c, 0x3f, 0xb5, 0xa8, 0x20, 0xcc, 0x74, 0x57, 0xd5, 0x9e,
				0xd2, 0x3a, 0x23, 0xa2, 0x3d, 0x63, 0xcd, 0x23, 0x04, 0x94, 0xc9, 0x63, 0x99, 0xef, 0xd5, 0x66, 0x71,
				0x0d, 0x46, 0x2e, 0x40, 0xba, 0x36, 0x56, 0x2f, 0x1b, 0x71, 0xf0, 0x62, 0x6c, 0xa7, 0x42, 0xfe, 0xa8,
				0x17, 0x01, 0xaf, 0xfc, 0xa1, 0x0b, 0x4b, 0x0e, 0xd9, 0x49, 0xda, 0xdb, 0x4d, 0x0d, 0x07, 0x5e, 0xf6,
				0x5b, 0xa8, 0xc5, 0x08, 0xec, 0x16, 0x8c, 0xb2, 0x49, 0xaf, 0x82, 0x6d, 0x46, 0xee, 0x82, 0x99, 0xd5,
				0x88, 0x85, 0xec, 0xef, 0x62, 0xa1, 0x53, 0x5c, 0xd3, 0xee, 0xc0, 0x49, 0xba, 0xa6, 0x64, 0xde, 0xd9,
				0xf7, 0xc1, 0x06, 0x53, 0xf4, 0x21, 0xd8, 0xaf, 0xc1, 0x81, 0x47, 0xbc, 0x1e, 0xcd, 0x17, 0x55, 0xc7,
				0x4f, 0x2a, 0xbb, 0x72, 0x62, 0x7a, 0x10, 0x1d, 0xdd, 0xb2, 0x9c, 0xa3, 0xdc, 0x30, 0xc9, 0x53, 0x12,
				0x28, 0x76, 0xff, 0x61, 0xc3, 0x1e, 0x34, 0x4f, 0x27, 0x66, 0xb2, 0xc0, 0x8a, 0x4a, 0x36, 0x7b, 0xf8,
				0xa0, 0xfa, 0x3f
			},
			.public_key_size = 258,
			.key_tag = 30909,
			.is_zone_key = true,
			.is_secure_entry_point = true,
			.has_supported_algorithm = true,
			.is_valid_for_dnssec = true,
			.priority = 3,
		}
	},

	//==================================================================================================================
	// MARK: - NSEC3
	// NSEC3 from com.
	{
		.name = {
			32, 'c', 'k', '0', 'p', 'o', 'j', 'm', 'g', '8', '7', '4', 'l', 'j', 'r', 'e', 'f', '7', 'e', 'f', 'n', '8',
			'4', '3', '0', 'q', 'v', 'i', 't', '8', 'b', 's', 'm', 3, 'c', 'o', 'm', 0
		},
		.type = kDNSRecordType_NSEC3,
		.class = kDNSClassType_IN,
		.rdata_len = 35,
		.rdata = {
			0x01, 0x01, 0x00, 0x00, 0x00, 0x14, 0x65, 0x01, 0xa0, 0xc2, 0x57, 0x20, 0xee, 0x15, 0x6f, 0x6c, 0x4e, 0x39,
			0x63, 0x6b, 0x3a, 0xda, 0x03, 0x12, 0xd9, 0x2a, 0x00, 0x07, 0x22, 0x00, 0x00, 0x00, 0x00, 0x02, 0x90
		},
		.expected_result_u.nsec3 = {
			.current_owner_name = {
				32, 'c', 'k', '0', 'p', 'o', 'j', 'm', 'g', '8', '7', '4', 'l', 'j', 'r', 'e', 'f', '7', 'e', 'f', 'n',
				'8', '4', '3', '0', 'q', 'v', 'i', 't', '8', 'b', 's', 'm', 3, 'c', 'o', 'm', 0
			},
			.hash_algorithm = NSEC3_HASH_ALGORITHM_SHA_1,
			.flags = NSEC3_FLAG_OPT_OUT,
			.iterations = 0,
			.salt_length = 0,
			.salt = {0},
			.hash_length = 20,
			.next_hashed_owner_name_in_binary = {
				0x65, 0x01, 0xa0, 0xc2, 0x57, 0x20, 0xee, 0x15, 0x6f, 0x6c, 0x4e, 0x39, 0x63, 0x6b, 0x3a, 0xda, 0x03,
				0x12, 0xd9, 0x2a
			},
			.next_hashed_owner_name = {
				32, 'C', 'K', '0', 'Q', '1', 'G', 'I', 'N', '4', '3', 'N', '1', 'A', 'R', 'R', 'C', '9', 'O', 'S', 'M',
				'6', 'Q', 'P', 'Q', 'R', '8', '1', 'H', '5', 'M', '9', 'A', 3, 'c', 'o', 'm', 0
			},
			.types_covered = {
				kDNSRecordType_NS,
				kDNSRecordType_SOA,
				kDNSRecordType_RRSIG,
				kDNSRecordType_DNSKEY,
				kDNSRecordType_NSEC3PARAM,
				kDNSRecordType_Invalid
			}
		}
	},
	{
		.name = {
			32, 's', '0', 'n', 'u', '3', '5', 's', 'd', 'b', '7', 'k', '3', 'j', '8', 'q', '3', 't', 'j', 'l', 'v', '6',
			'8', '7', '4', 'a', 'j', '0', '5', '8', 'i', '1', 'o', 3, 'c', 'o', 'm', 0
		},
		.type = kDNSRecordType_NSEC3,
		.class = kDNSClassType_IN,
		.rdata_len = 34,
		.rdata = {
			0x01, 0x01, 0x00, 0x00, 0x00, 0x14, 0xe0, 0x2f, 0xe4, 0xd9, 0xec, 0xf8, 0x74, 0x1b, 0xf1, 0x76, 0x56, 0x75,
			0x6f, 0x02, 0x27, 0xb4, 0x6e, 0x43, 0xcd, 0xbd, 0x00, 0x06, 0x20, 0x00, 0x00, 0x00, 0x00, 0x12
		},
		.expected_result_u.nsec3 = {
			.current_owner_name = {
				32, 's', '0', 'n', 'u', '3', '5', 's', 'd', 'b', '7', 'k', '3', 'j', '8', 'q', '3', 't', 'j', 'l', 'v',
				'6', '8', '7', '4', 'a', 'j', '0', '5', '8', 'i', '1', 'o', 3, 'c', 'o', 'm', 0
			},
			.hash_algorithm = NSEC3_HASH_ALGORITHM_SHA_1,
			.flags = NSEC3_FLAG_OPT_OUT,
			.iterations = 0,
			.salt_length = 0,
			.salt = {0},
			.hash_length = 20,
			.next_hashed_owner_name_in_binary = {
				0xe0, 0x2f, 0xe4, 0xd9, 0xec, 0xf8, 0x74, 0x1b, 0xf1, 0x76, 0x56, 0x75, 0x6f, 0x02, 0x27, 0xb4, 0x6e,
				0x43, 0xcd, 0xbd
			},
			.next_hashed_owner_name = {
				32, 'S', '0', 'N', 'U', '9', 'M', 'F', 'C', 'V', '1', 'Q', '1', 'N', 'S', 'B', 'M', 'A', 'P', 'Q', 'M',
				'U', '0', 'H', '7', 'M', 'H', 'N', '4', '7', 'J', 'D', 'T', 3, 'c', 'o', 'm', 0
			},
			.types_covered = {
				kDNSRecordType_NS,
				kDNSRecordType_DS,
				kDNSRecordType_RRSIG,
				kDNSRecordType_Invalid
			}
		}
	},
	// NSEC3 from gov.
	{
		.name = {
			32, '5', '7', '8', 'E', 'T', '1', '6', 'S', '7', 'L', 'T', 'N', 'S', 'Q', '1', 'T', '0', 'A', 'M', 'M', '2',
			'1', 'G', 'L', '2', '0', 'O', 'J', '5', 'G', '7', '6', 3, 'g', 'o', 'v', 0
		},
		.type = kDNSRecordType_NSEC3,
		.class = kDNSClassType_IN,
		.rdata_len = 41,
		.rdata = {
			0x01, 0x00, 0x00, 0x08, 0x06, 0x4c, 0x44, 0x93, 0x48, 0x02, 0xd3, 0x14, 0x29, 0xd1, 0x12, 0xdf, 0xa2, 0x74,
			0xb5, 0x11, 0x74, 0xcc, 0x77, 0x6b, 0x77, 0x2e, 0xfd, 0xa9, 0x45, 0xd0, 0x3b, 0xa8, 0x00, 0x07, 0x22, 0x00,
			0x00, 0x00, 0x00, 0x02, 0x90
		},
		.expected_result_u.nsec3 = {
			.current_owner_name = {
				32, '5', '7', '8', 'E', 'T', '1', '6', 'S', '7', 'L', 'T', 'N', 'S', 'Q', '1', 'T', '0', 'A', 'M', 'M',
				'2', '1', 'G', 'L', '2', '0', 'O', 'J', '5', 'G', '7', '6', 3, 'g', 'o', 'v', 0
			},
			.hash_algorithm = NSEC3_HASH_ALGORITHM_SHA_1,
			.flags = 0,
			.iterations = 8,
			.salt_length = 6,
			.salt = {
				0x4c, 0x44, 0x93, 0x48, 0x02, 0xd3
			},
			.hash_length = 20,
			.next_hashed_owner_name_in_binary = {
				0x29, 0xd1, 0x12, 0xdf, 0xa2, 0x74, 0xb5, 0x11, 0x74, 0xcc, 0x77, 0x6b, 0x77, 0x2e, 0xfd, 0xa9, 0x45,
				0xd0, 0x3b, 0xa8
			},
			.next_hashed_owner_name = {
				32, '5', '7', '8', 'H', '5', 'N', 'T', '2', 'E', 'I', 'Q', 'H', '2', 'T', '6', 'C', 'E', 'T', 'L', 'N',
				'E', 'B', 'N', 'T', 'L', '5', '2', 'T', '0', 'E', 'T', '8', 3, 'g', 'o', 'v', 0
			},
			.types_covered = {
				kDNSRecordType_NS,
				kDNSRecordType_SOA,
				kDNSRecordType_RRSIG,
				kDNSRecordType_DNSKEY,
				kDNSRecordType_NSEC3PARAM,
				kDNSRecordType_Invalid
			}
		}
	},
	{
		.name = {
			32, 'k', 'a', '6', '9', '0', 'f', 'n', 'a', 't', 'n', 'i', 'q', 'i', 'd', '7', '8', 'p', 'v', 'j', 'h', '8',
			'b', '2', '9', 't', 'b', '4', 'p', '3', 'b', 'i', 'o', 3, 'g', 'o', 'v', 0
		},
		.type = kDNSRecordType_NSEC3,
		.class = kDNSClassType_IN,
		.rdata_len = 35,
		.rdata = {
			0x01, 0x00, 0x00, 0x08, 0x06, 0x4c, 0x44, 0x93, 0x48, 0x02, 0xd3, 0x14, 0xa2, 0xb2, 0xf0, 0x36, 0x3f, 0x82,
			0xb9, 0xe2, 0x3b, 0x04, 0xb7, 0xcd, 0xc2, 0xd3, 0x25, 0x51, 0x53, 0x7a, 0xac, 0xfa, 0x00, 0x01, 0x20
		},
		.expected_result_u.nsec3 = {
			.current_owner_name = {
				32, 'k', 'a', '6', '9', '0', 'f', 'n', 'a', 't', 'n', 'i', 'q', 'i', 'd', '7', '8', 'p', 'v', 'j', 'h',
				'8', 'b', '2', '9', 't', 'b', '4', 'p', '3', 'b', 'i', 'o', 3, 'g', 'o', 'v', 0
			},
			.hash_algorithm = NSEC3_HASH_ALGORITHM_SHA_1,
			.flags = 0,
			.iterations = 8,
			.salt_length = 6,
			.salt = {
				0x4c, 0x44, 0x93, 0x48, 0x02, 0xd3
			},
			.hash_length = 20,
			.next_hashed_owner_name_in_binary = {
				0xa2, 0xb2, 0xf0, 0x36, 0x3f, 0x82, 0xb9, 0xe2, 0x3b, 0x04, 0xb7, 0xcd, 0xc2, 0xd3, 0x25, 0x51, 0x53,
				0x7a, 0xac, 0xfa

			},
			.next_hashed_owner_name = {
				32, 'K', 'A', 'P', 'F', '0', 'D', 'H', 'V', 'G', 'A', 'S', 'U', '4', 'E', 'O', '4', 'M', 'V', '6', 'S',
				'5', 'K', 'P', '5', 'A', '5', '9', 'N', 'L', 'B', '7', 'Q', 3, 'g', 'o', 'v', 0
			},
			.types_covered = {
				kDNSRecordType_NS,
				kDNSRecordType_Invalid
			}
		}
	},
	{
		.name = {
			32, 'u', 'f', 'q', 'f', 'b', '8', 'f', 'a', '0', '3', 'k', 'm', 'u', 'c', '5', 'c', 'g', '9', 'u', 'a', 's',
			's', 's', 'n', 'o', 'g', 'n', 't', '6', 'g', 'b', '9', 3, 'g', 'o', 'v', 0
		},
		.type = kDNSRecordType_NSEC3,
		.class = kDNSClassType_IN,
		.rdata_len = 35,
		.rdata = {
			0x01, 0x00, 0x00, 0x08, 0x06, 0x4c, 0x44, 0x93, 0x48, 0x02, 0xd3, 0x14, 0xf4, 0x13, 0xd5, 0x8a, 0xa6, 0x6b,
			0xa4, 0x88, 0xdb, 0xb4, 0x87, 0x39, 0xf6, 0x8c, 0x3a, 0xf9, 0xa9, 0x02, 0x33, 0x00, 0x00, 0x01, 0x20
		},
		.expected_result_u.nsec3 = {
			.current_owner_name = {
				32, 'u', 'f', 'q', 'f', 'b', '8', 'f', 'a', '0', '3', 'k', 'm', 'u', 'c', '5', 'c', 'g', '9', 'u', 'a',
				's', 's', 's', 'n', 'o', 'g', 'n', 't', '6', 'g', 'b', '9', 3, 'g', 'o', 'v', 0
			},
			.hash_algorithm = NSEC3_HASH_ALGORITHM_SHA_1,
			.flags = 0,
			.iterations = 8,
			.salt_length = 6,
			.salt = {
				0x4c, 0x44, 0x93, 0x48, 0x02, 0xd3
			},
			.hash_length = 20,
			.next_hashed_owner_name_in_binary = {
				0xf4, 0x13, 0xd5, 0x8a, 0xa6, 0x6b, 0xa4, 0x88, 0xdb, 0xb4, 0x87, 0x39, 0xf6, 0x8c, 0x3a, 0xf9, 0xa9,
				0x02, 0x33, 0x00
			},
			.next_hashed_owner_name = {
				32, 'U', 'G', '9', 'T', 'B', '2', 'L', '6', 'D', 'E', 'I', '8', 'H', 'M', 'T', 'K', 'G', 'S', 'S', 'V',
				'D', '3', '1', 'Q', 'V', '6', 'K', 'G', '4', 'C', 'O', '0', 3, 'g', 'o', 'v', 0
			},
			.types_covered = {
				kDNSRecordType_NS,
				kDNSRecordType_Invalid
			}
		}
	}
};

- (void)testCreate
{
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		dns_obj_error_t err;
		dns_obj_rr_t rr1 = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		dns_obj_rr_t rr2 = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, false, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		XCTAssertTrue(dns_obj_equal(rr1, rr2));

		MDNS_DISPOSE_DNS_OBJ(rr1);
		MDNS_DISPOSE_DNS_OBJ(rr2);
	}
}

- (void)testGetName
{
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		dns_obj_error_t err;
		dns_obj_rr_t rr_allocated = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		dns_obj_rr_t rr = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, false, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		XCTAssertTrue(dns_obj_equal(rr_allocated, rr));

		dns_obj_domain_name_t expected_name = dns_obj_domain_name_create_with_labels(test_cases_rr[i].name, false, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		XCTAssertTrue(dns_obj_equal(expected_name, dns_obj_rr_get_name(rr_allocated)));
		XCTAssertTrue(dns_obj_equal(expected_name, dns_obj_rr_get_name(rr)));

		XCTAssert(dns_obj_rr_get_name_in_labels(rr_allocated) != test_cases_rr[i].name);
		if (domain_name_labels_contains_upper_case(test_cases_rr[i].name))
		{
			XCTAssert(dns_obj_rr_get_name_in_labels(rr) != test_cases_rr[i].name);
		}
		else
		{
			XCTAssert(dns_obj_rr_get_name_in_labels(rr) == test_cases_rr[i].name);
		}

		MDNS_DISPOSE_DNS_OBJ(expected_name);
		MDNS_DISPOSE_DNS_OBJ(rr);
		MDNS_DISPOSE_DNS_OBJ(rr_allocated);
	}
}

- (void)testGetType
{
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		dns_obj_error_t err;
		dns_obj_rr_t rr = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		XCTAssertEqual(dns_obj_rr_get_type(rr), test_cases_rr[i].type);
		MDNS_DISPOSE_DNS_OBJ(rr);
	}
}

- (void)testGetClass
{
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		dns_obj_error_t err;
		dns_obj_rr_t rr = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		XCTAssertEqual(dns_obj_rr_get_class(rr), test_cases_rr[i].class);
		MDNS_DISPOSE_DNS_OBJ(rr);
	}
}

- (void)testGetRdataLen
{
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		dns_obj_error_t err;
		dns_obj_rr_t rr = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		XCTAssertEqual(dns_obj_rr_get_rdata_len(rr), test_cases_rr[i].rdata_len);
		MDNS_DISPOSE_DNS_OBJ(rr);
	}
}

- (void)testGetRdata
{
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		dns_obj_error_t err;
		dns_obj_rr_t rr_allocated = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		dns_obj_rr_t rr = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, false, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);
		XCTAssertTrue(dns_obj_equal(rr_allocated, rr));

		XCTAssertEqual(dns_obj_rr_get_rdata_len(rr_allocated), test_cases_rr[i].rdata_len);
		XCTAssertEqual(dns_obj_rr_get_rdata_len(rr), test_cases_rr[i].rdata_len);

		XCTAssertEqual(memcmp(dns_obj_rr_get_rdata(rr_allocated), dns_obj_rr_get_rdata(rr), test_cases_rr[i].rdata_len), 0);

		XCTAssert(dns_obj_rr_get_rdata(rr_allocated) != test_cases_rr[i].rdata);
		XCTAssert(dns_obj_rr_get_rdata(rr) == test_cases_rr[i].rdata);

		MDNS_DISPOSE_DNS_OBJ(rr);
		MDNS_DISPOSE_DNS_OBJ(rr_allocated);
	}
}

- (void)testSignedData
{
#define MAX_RDATA_LEN 264
	typedef struct test_signed_data_s {
		const resource_record_bytes_short_t	record;
		const uint32_t						original_ttl;
		const uint8_t						rrsig_labels;
		const uint8_t						expected_signed_data[MAX_RDATA_LEN];
	} test_signed_data_t;

	const test_signed_data_t test_cases_rr[] = {
		{
			.record = {
				.name = {
					3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0
				},
				.class = kDNSClassType_IN,
				.type = kDNSRecordType_AAAA,
				.rdata_len = 16,
				.rdata = {
					0x26, 0x06, 0x47, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x10, 0x7b, 0x60
				}
			},
			.original_ttl = 300,
			.rrsig_labels = 3,
			.expected_signed_data = {
				3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0,
				0x00, 0x1c, 0x00, 0x01, 0x00, 0x00, 0x01, 0x2c, 0x00, 0x10, 0x26, 0x06, 0x47, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x10, 0x7b, 0x60
			},
		},
		{
			.record = {
				.name = {
					3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0
				},
				.class = kDNSClassType_IN,
				.type = kDNSRecordType_DNSKEY,
				.rdata_len = 68,
				.rdata = {
					0x01, 0x00, 0x03, 0x0d, 0xa0, 0x93, 0x11, 0x11, 0x2c, 0xf9, 0x13, 0x88, 0x18, 0xcd, 0x2f, 0xea,
					0xe9, 0x70, 0xeb, 0xbd, 0x4d, 0x6a, 0x30, 0xf6, 0x08, 0x8c, 0x25, 0xb3, 0x25, 0xa3, 0x9a, 0xbb,
					0xc5, 0xcd, 0x11, 0x97, 0xaa, 0x09, 0x82, 0x83, 0xe5, 0xaa, 0xf4, 0x21, 0x17, 0x7c, 0x2a, 0xa5,
					0xd7, 0x14, 0x99, 0x2a, 0x99, 0x57, 0xd1, 0xbc, 0xc1, 0x8f, 0x98, 0xcd, 0x71, 0xf1, 0xf1, 0x80,
					0x6b, 0x65, 0xe1, 0x48
				},
			},
			.original_ttl = 3600,
			.rrsig_labels = 3,
			.expected_signed_data = {
				3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0,
				0x00, 0x30, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x44, 0x01, 0x00, 0x03, 0x0d, 0xa0, 0x93, 0x11,
				0x11, 0x2c, 0xf9, 0x13, 0x88, 0x18, 0xcd, 0x2f, 0xea, 0xe9, 0x70, 0xeb, 0xbd, 0x4d, 0x6a, 0x30, 0xf6,
				0x08, 0x8c, 0x25, 0xb3, 0x25, 0xa3, 0x9a, 0xbb, 0xc5, 0xcd, 0x11, 0x97, 0xaa, 0x09, 0x82, 0x83, 0xe5,
				0xaa, 0xf4, 0x21, 0x17, 0x7c, 0x2a, 0xa5, 0xd7, 0x14, 0x99, 0x2a, 0x99, 0x57, 0xd1, 0xbc, 0xc1, 0x8f,
				0x98, 0xcd, 0x71, 0xf1, 0xf1, 0x80, 0x6b, 0x65, 0xe1, 0x48
			}
		},
		{
			.record = {
				.name = {
					8, 'w', 'i', 'l', 'd', 'c', 'a', 'r', 'd', 9, 's', 'u', 'b', 'd', 'o', 'm', 'a', 'i', 'n', 1, 'd',
					5, 'q', 'd', 'e', 'n', 'g', 2, 'i', 'o', 0
				},
				.class = kDNSClassType_IN,
				.type = kDNSRecordType_NSEC,
				.rdata_len = 18,
				.rdata = {
					0x05, 0x71, 0x64, 0x65, 0x6e, 0x67, 0x02, 0x69, 0x6f, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08,
					0x00, 0x03
				},
			},
			.original_ttl = 3601,
			.rrsig_labels = 3,
			.expected_signed_data = {
				1, '*', 1, 'd', 5, 'q', 'd', 'e', 'n', 'g', 2, 'i', 'o', 0, 0x00, 0x2f, 0x00, 0x01, 0x00, 0x00, 0x0e,
				0x11, 0x00, 0x12, 0x05, 0x71, 0x64, 0x65, 0x6e, 0x67, 0x02, 0x69, 0x6f, 0x00, 0x00, 0x06, 0x00, 0x00,
				0x00, 0x08, 0x00, 0x03
			}
		},
	};

	dns_obj_error_t err;
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		const resource_record_bytes_short_t * const record_bytes = &test_cases_rr[i].record;
		dns_obj_rr_t rr = dns_obj_rr_create(record_bytes->name, record_bytes->type, record_bytes->class,
			record_bytes->rdata, record_bytes->rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		XCTAssertEqual(dns_obj_rr_get_signed_data(rr), NULL);
		XCTAssertEqual(dns_obj_rr_get_signed_data_len(rr), 0);

		dns_obj_rr_set_comparison_attributes(rr, test_cases_rr[i].original_ttl, test_cases_rr[i].rrsig_labels);

		XCTAssertEqual(memcmp(dns_obj_rr_get_signed_data(rr), test_cases_rr[i].expected_signed_data,
							  dns_obj_rr_get_signed_data_len(rr)),
					   0);

		dns_obj_rr_clear_comparison_attributes(rr);

		dns_obj_rr_set_comparison_attributes(rr, test_cases_rr[i].original_ttl, test_cases_rr[i].rrsig_labels);

		XCTAssertEqual(memcmp(dns_obj_rr_get_signed_data(rr), test_cases_rr[i].expected_signed_data,
							  dns_obj_rr_get_signed_data_len(rr)),
					   0);

		MDNS_DISPOSE_DNS_OBJ(rr);
	}
}

- (void)testEqualToRawData
{
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		for (size_t j = 0; j < countof(test_cases_rr); j++) {
			dns_obj_error_t err;
			dns_obj_rr_t rr_i = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
				test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, ((i % 2) == 0), &err);
			XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

			const resource_record_bytes_t * const raw_rr_j = &test_cases_rr[j];
			const bool equal = dns_obj_rr_equal_to_raw_data(rr_i, raw_rr_j->name, raw_rr_j->type, raw_rr_j->class,
				raw_rr_j->rdata, raw_rr_j->rdata_len);
			XCTAssertEqual(equal, i == j);

			MDNS_DISPOSE_DNS_OBJ(rr_i);
		}
	}
}

- (void)testBelongToOneRRSet
{
	dns_obj_error_t err;

	const resource_record_bytes_short_t same_set[] = {
		{
			.name = {
				3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_AAAA,
			.rdata_len = 16,
			.rdata = {
				0x26, 0x06, 0x47, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x10, 0x7b, 0x60
			}
		},
		{
			.name = {
				3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_AAAA,
			.rdata_len = 16,
			.rdata = {
				0x26, 0x06, 0x47, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x10, 0x7c, 0x60
			}
		},
	};

	dns_obj_rr_t rrs_in_same_set[countof(same_set)] = {NULL};

	for (size_t i = 0; i < countof(same_set); i++) {
		rrs_in_same_set[i] = dns_obj_rr_create(same_set[i].name, same_set[i].type, same_set[i].class,
			same_set[i].rdata, same_set[i].rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);
	}

	XCTAssertTrue(dns_obj_rrs_belong_to_one_rrset(rrs_in_same_set, countof(rrs_in_same_set)));


	const resource_record_bytes_short_t not_same_set[] = {
		{
			.name = {
				3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_AAAA,
			.rdata_len = 16,
			.rdata = {
				0x26, 0x06, 0x47, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x10, 0x7b, 0x60
			}
		},
		{
			.name = {
				3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DNSKEY,
			.rdata_len = 68,
			.rdata = {
				0x01, 0x00, 0x03, 0x0d, 0xa0, 0x93, 0x11, 0x11, 0x2c, 0xf9, 0x13, 0x88, 0x18, 0xcd, 0x2f, 0xea,
				0xe9, 0x70, 0xeb, 0xbd, 0x4d, 0x6a, 0x30, 0xf6, 0x08, 0x8c, 0x25, 0xb3, 0x25, 0xa3, 0x9a, 0xbb,
				0xc5, 0xcd, 0x11, 0x97, 0xaa, 0x09, 0x82, 0x83, 0xe5, 0xaa, 0xf4, 0x21, 0x17, 0x7c, 0x2a, 0xa5,
				0xd7, 0x14, 0x99, 0x2a, 0x99, 0x57, 0xd1, 0xbc, 0xc1, 0x8f, 0x98, 0xcd, 0x71, 0xf1, 0xf1, 0x80,
				0x6b, 0x65, 0xe1, 0x48
			},
		},
	};

	dns_obj_rr_t rrs_in_different_set[countof(not_same_set)] = {NULL};

	for (size_t i = 0; i < countof(not_same_set); i++) {
		rrs_in_different_set[i] = dns_obj_rr_create(not_same_set[i].name, not_same_set[i].type, not_same_set[i].class,
			not_same_set[i].rdata, not_same_set[i].rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);
	}

	XCTAssertFalse(dns_obj_rrs_belong_to_one_rrset(rrs_in_different_set, countof(rrs_in_different_set)));

	for (size_t i = 0; i < countof(rrs_in_same_set); i++) {
		MDNS_DISPOSE_DNS_OBJ(rrs_in_same_set[i]);
	}
	for (size_t i = 0; i < countof(rrs_in_different_set); i++) {
		MDNS_DISPOSE_DNS_OBJ(rrs_in_different_set[i]);
	}
}

- (void)testCompareEquality
{
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		for (size_t j = 0; j < countof(test_cases_rr); j++) {
			dns_obj_error_t err;
			dns_obj_rr_t rr_i = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
				test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, ((i % 2) == 0), &err);
			XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

			dns_obj_rr_t rr_j = dns_obj_rr_create(test_cases_rr[j].name, test_cases_rr[j].type, test_cases_rr[j].class,
				test_cases_rr[j].rdata, test_cases_rr[j].rdata_len, ((j % 2) == 0), &err);
			XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

			XCTAssertTrue(dns_obj_equal(rr_i, rr_i));
			XCTAssertTrue(dns_obj_equal(rr_j, rr_j));

			if (i == j) {
				XCTAssertTrue(dns_obj_equal(rr_i, rr_j));
				XCTAssertEqual(dns_obj_compare(rr_i, rr_j), compare_result_equal);
			} else {
				XCTAssertFalse(dns_obj_equal(rr_i, rr_j));
			}

			MDNS_DISPOSE_DNS_OBJ(rr_j);
			MDNS_DISPOSE_DNS_OBJ(rr_i);
		}
	}
}

- (void)testGetSignedData
{
#define MAX_RDATA_LEN 264
	typedef struct test_signed_data_s {
		const resource_record_bytes_short_t	record;
		const uint32_t						original_ttl;
		const uint8_t						rrsig_labels;
		const uint8_t						expected_signed_data[MAX_RDATA_LEN];
	} test_signed_data_t;

const test_signed_data_t test_cases_rr[] = {
		{
			.record = {
				.name = {
					8, 'w', 'i', 'l', 'd', 'c', 'a', 'r', 'd', 9, 's', 'u', 'b', 'd', 'o', 'm', 'a', 'i', 'n', 1, 'd',
					5, 'q', 'd', 'e', 'n', 'g', 2, 'i', 'o', 0
				},
				.class = kDNSClassType_IN,
				.type = kDNSRecordType_NSEC,
				.rdata_len = 18,
				.rdata = {
					0x05, 0x71, 0x64, 0x65, 0x6e, 0x67, 0x02, 0x69, 0x6f, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08,
					0x00, 0x03
				},
			},
			.original_ttl = 3601,
			.rrsig_labels = 3,
			.expected_signed_data = {
				1, '*', 1, 'd', 5, 'q', 'd', 'e', 'n', 'g', 2, 'i', 'o', 0, 0x00, 0x2f, 0x00, 0x01, 0x00, 0x00, 0x0e,
				0x11, 0x00, 0x12, 0x05, 0x71, 0x64, 0x65, 0x6e, 0x67, 0x02, 0x69, 0x6f, 0x00, 0x00, 0x06, 0x00, 0x00,
				0x00, 0x08, 0x00, 0x03
			}
		},
		{
			.record = {
				.name = {
					3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0
				},
				.class = kDNSClassType_IN,
				.type = kDNSRecordType_AAAA,
				.rdata_len = 16,
				.rdata = {
					0x26, 0x06, 0x47, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x10, 0x7b, 0x60
				}
			},
			.original_ttl = 300,
			.rrsig_labels = 3,
			.expected_signed_data = {
				3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0,
				0x00, 0x1c, 0x00, 0x01, 0x00, 0x00, 0x01, 0x2c, 0x00, 0x10, 0x26, 0x06, 0x47, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x10, 0x7b, 0x60
			},
		},
		{
			.record = {
				.name = {
					3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0
				},
				.class = kDNSClassType_IN,
				.type = kDNSRecordType_DNSKEY,
				.rdata_len = 68,
				.rdata = {
					0x01, 0x00, 0x03, 0x0d, 0xa0, 0x93, 0x11, 0x11, 0x2c, 0xf9, 0x13, 0x88, 0x18, 0xcd, 0x2f, 0xea,
					0xe9, 0x70, 0xeb, 0xbd, 0x4d, 0x6a, 0x30, 0xf6, 0x08, 0x8c, 0x25, 0xb3, 0x25, 0xa3, 0x9a, 0xbb,
					0xc5, 0xcd, 0x11, 0x97, 0xaa, 0x09, 0x82, 0x83, 0xe5, 0xaa, 0xf4, 0x21, 0x17, 0x7c, 0x2a, 0xa5,
					0xd7, 0x14, 0x99, 0x2a, 0x99, 0x57, 0xd1, 0xbc, 0xc1, 0x8f, 0x98, 0xcd, 0x71, 0xf1, 0xf1, 0x80,
					0x6b, 0x65, 0xe1, 0x48
				},
			},
			.original_ttl = 3600,
			.rrsig_labels = 3,
			.expected_signed_data = {
				3, 'w', 'w', 'w', 10, 'c', 'l', 'o', 'u', 'd', 'f', 'l', 'a', 'r', 'e', 3, 'c', 'o', 'm', 0,
				0x00, 0x30, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x44, 0x01, 0x00, 0x03, 0x0d, 0xa0, 0x93, 0x11,
				0x11, 0x2c, 0xf9, 0x13, 0x88, 0x18, 0xcd, 0x2f, 0xea, 0xe9, 0x70, 0xeb, 0xbd, 0x4d, 0x6a, 0x30, 0xf6,
				0x08, 0x8c, 0x25, 0xb3, 0x25, 0xa3, 0x9a, 0xbb, 0xc5, 0xcd, 0x11, 0x97, 0xaa, 0x09, 0x82, 0x83, 0xe5,
				0xaa, 0xf4, 0x21, 0x17, 0x7c, 0x2a, 0xa5, 0xd7, 0x14, 0x99, 0x2a, 0x99, 0x57, 0xd1, 0xbc, 0xc1, 0x8f,
				0x98, 0xcd, 0x71, 0xf1, 0xf1, 0x80, 0x6b, 0x65, 0xe1, 0x48
			}
		},
	};

	dns_obj_error_t err;
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		const resource_record_bytes_short_t * const record_bytes = &test_cases_rr[i].record;
		dns_obj_rr_t rr = dns_obj_rr_create(record_bytes->name, record_bytes->type, record_bytes->class, record_bytes->rdata,
			record_bytes->rdata_len, true, &err);
		XCTAssertEqual(err, DNS_OBJ_ERROR_NO_ERROR);

		const uint8_t * const null_signed_data = dns_obj_rr_get_signed_data(rr);
		XCTAssertEqual(null_signed_data, NULL);

		dns_obj_rr_set_comparison_attributes(rr, test_cases_rr[i].original_ttl, test_cases_rr[i].rrsig_labels);
		const uint8_t * const signed_data = dns_obj_rr_get_signed_data(rr);
		XCTAssertNotEqual(signed_data, NULL);

		const int reslt = memcmp(signed_data, test_cases_rr[i].expected_signed_data, dns_obj_rr_get_signed_data_len(rr));
		XCTAssertEqual(reslt, 0);

		dns_obj_rr_clear_comparison_attributes(rr);
		const uint8_t * const null_signed_data_again = dns_obj_rr_get_signed_data(rr);
		XCTAssertEqual(null_signed_data_again, NULL);

		MDNS_DISPOSE_DNS_OBJ(rr);
	}
}

- (void)testCompareBasic
{
	const resource_record_bytes_short_t smaller = {
		.name = {
			7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
		},
		.class = kDNSClassType_IN,
		.type = kDNSRecordType_DS,
		.rdata_len = 36,
		.rdata = {
			0x0d, 0x45, 0x08, 0x02, 0xed, 0x11, 0x68, 0x60, 0x4b, 0xc6, 0xa1, 0x40, 0x68, 0xb9, 0x90, 0x54, 0x01, 0xe6,
			0x26, 0x98, 0xbb, 0x36, 0x63, 0xb6, 0xec, 0x20, 0x73, 0xeb, 0xd3, 0x59, 0x9b, 0x88, 0x2a, 0x78, 0x5b, 0xf6
		},
	};

	const resource_record_bytes_short_t bigger = {
		.name = {
			7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
		},
		.class = kDNSClassType_IN,
		.type = kDNSRecordType_DS,
		.rdata_len = 24,
		.rdata = {
			0x93, 0x94, 0x08, 0x01, 0xb4, 0xa5, 0xcc, 0xe8, 0xd8, 0x2d, 0xc5, 0x85, 0xe3, 0x27, 0xe5, 0x89, 0x6e, 0xae,
			0x82, 0xe0, 0xb9, 0xa7, 0x6d, 0xc6
		},
	};

	dns_obj_rr_t smaller_rr = dns_obj_rr_create(smaller.name, smaller.type, smaller.class, smaller.rdata, smaller.rdata_len, false, NULL);
	dns_obj_rr_t bigger_rr = dns_obj_rr_create(bigger.name, bigger.type, bigger.class, bigger.rdata, bigger.rdata_len, false, NULL);
	XCTAssertTrue(dns_obj_compare(smaller_rr, bigger_rr) == compare_result_less);

	MDNS_DISPOSE_DNS_OBJ(smaller_rr);
	MDNS_DISPOSE_DNS_OBJ(bigger_rr);
}

- (void)testSort
{
	const resource_record_bytes_short_t test_cases_rr[] = {
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 36,
			.rdata = {
				0x0d, 0x45, 0x08, 0x02, 0xed, 0x11, 0x68, 0x60, 0x4b, 0xc6, 0xa1, 0x40, 0x68, 0xb9, 0x90, 0x54, 0x01,
				0xe6, 0x26, 0x98, 0xbb, 0x36, 0x63, 0xb6, 0xec, 0x20, 0x73, 0xeb, 0xd3, 0x59, 0x9b, 0x88, 0x2a, 0x78,
				0x5b, 0xf6
			}
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 24,
			.rdata = {
				0x93, 0x94, 0x08, 0x01, 0xb4, 0xa5, 0xcc, 0xe8, 0xd8, 0x2d, 0xc5, 0x85, 0xe3, 0x27, 0xe5, 0x89, 0x6e,
				0xae, 0x82, 0xe0, 0xb9, 0xa7, 0x6d, 0xc6
			},
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 36,
			.rdata = {
				0x93, 0x94, 0x08, 0x02, 0xd9, 0x6a, 0xfa, 0x90, 0x22, 0x00, 0x0d, 0x36, 0x8b, 0x5f, 0x49, 0x78, 0x77,
				0xdf, 0x28, 0x9a, 0x1e, 0x9a, 0x13, 0xa1, 0xab, 0x1f, 0x97, 0xbc, 0x1b, 0xf4, 0xd5, 0xde, 0x16, 0x87,
				0x91, 0x34
			},
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 36,
			.rdata = {
				0x7b, 0x65, 0x08, 0x02, 0x3f, 0xdc, 0x4c, 0x11, 0xfa, 0x3a, 0xd3, 0x53, 0x5e, 0xa8, 0xc1, 0xce, 0x3e,
				0xaf, 0x7b, 0xfa, 0x5c, 0xa9, 0xae, 0x8a, 0x83, 0x4d, 0x98, 0xfe, 0xe1, 0x00, 0x85, 0xcf, 0xae, 0xb6,
				0x25, 0xaa
			},
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 24,
			.rdata = {
				0x7b, 0x65, 0x08, 0x01, 0x7b, 0x83, 0x70, 0x00, 0x28, 0x75, 0xdd, 0xa7, 0x81, 0x39, 0x0a, 0x8e, 0x58,
				0x6c, 0x31, 0x49, 0x38, 0x47, 0xd9, 0xbc
			},
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 24,
			.rdata = {
				0x0d, 0x45, 0x08, 0x01, 0xde, 0xe1, 0x03, 0x45, 0x94, 0x2c, 0x98, 0x71, 0x1e, 0xb0, 0x58, 0xb2, 0x5a,
				0x74, 0x9e, 0xe3, 0x42, 0xfc, 0xe1, 0xdc
			},
		},
	};

	const resource_record_bytes_short_t expected_order[] = {
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 24,
			.rdata = {
				0x0d, 0x45, 0x08, 0x01, 0xde, 0xe1, 0x03, 0x45, 0x94, 0x2c, 0x98, 0x71, 0x1e, 0xb0, 0x58, 0xb2, 0x5a,
				0x74, 0x9e, 0xe3, 0x42, 0xfc, 0xe1, 0xdc
			},
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 36,
			.rdata = {
				0x0d, 0x45, 0x08, 0x02, 0xed, 0x11, 0x68, 0x60, 0x4b, 0xc6, 0xa1, 0x40, 0x68, 0xb9, 0x90, 0x54, 0x01,
				0xe6, 0x26, 0x98, 0xbb, 0x36, 0x63, 0xb6, 0xec, 0x20, 0x73, 0xeb, 0xd3, 0x59, 0x9b, 0x88, 0x2a, 0x78,
				0x5b, 0xf6
			}
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 24,
			.rdata = {
				0x7b, 0x65, 0x08, 0x01, 0x7b, 0x83, 0x70, 0x00, 0x28, 0x75, 0xdd, 0xa7, 0x81, 0x39, 0x0a, 0x8e, 0x58,
				0x6c, 0x31, 0x49, 0x38, 0x47, 0xd9, 0xbc
			},
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 36,
			.rdata = {
				0x7b, 0x65, 0x08, 0x02, 0x3f, 0xdc, 0x4c, 0x11, 0xfa, 0x3a, 0xd3, 0x53, 0x5e, 0xa8, 0xc1, 0xce, 0x3e,
				0xaf, 0x7b, 0xfa, 0x5c, 0xa9, 0xae, 0x8a, 0x83, 0x4d, 0x98, 0xfe, 0xe1, 0x00, 0x85, 0xcf, 0xae, 0xb6,
				0x25, 0xaa
			},
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 24,
			.rdata = {
				0x93, 0x94, 0x08, 0x01, 0xb4, 0xa5, 0xcc, 0xe8, 0xd8, 0x2d, 0xc5, 0x85, 0xe3, 0x27, 0xe5, 0x89, 0x6e,
				0xae, 0x82, 0xe0, 0xb9, 0xa7, 0x6d, 0xc6
			},
		},
		{
			.name = {
				7, 'e', 'x', 'a', 'm', 'p', 'l', 'e', 3, 'o', 'r', 'g', 0
			},
			.class = kDNSClassType_IN,
			.type = kDNSRecordType_DS,
			.rdata_len = 36,
			.rdata = {
				0x93, 0x94, 0x08, 0x02, 0xd9, 0x6a, 0xfa, 0x90, 0x22, 0x00, 0x0d, 0x36, 0x8b, 0x5f, 0x49, 0x78, 0x77,
				0xdf, 0x28, 0x9a, 0x1e, 0x9a, 0x13, 0xa1, 0xab, 0x1f, 0x97, 0xbc, 0x1b, 0xf4, 0xd5, 0xde, 0x16, 0x87,
				0x91, 0x34
			},
		},
	};

	check_compile_time_code(countof(test_cases_rr) == countof(expected_order));

	dns_obj_rr_t rrs[countof(test_cases_rr)] = {NULL};
	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		rrs[i] = dns_obj_rr_create(test_cases_rr[i].name, test_cases_rr[i].type, test_cases_rr[i].class,
			test_cases_rr[i].rdata, test_cases_rr[i].rdata_len, true, NULL);
	}

	dns_objs_sort(rrs, countof(rrs), sort_order_ascending);

	for (size_t i = 0; i < countof(test_cases_rr); i++) {
		dns_obj_rr_t expect = dns_obj_rr_create(expected_order[i].name, expected_order[i].type, expected_order[i].class,
			expected_order[i].rdata, expected_order[i].rdata_len, true, NULL);

		XCTAssertTrue(dns_obj_equal(expect, rrs[i]));

		MDNS_DISPOSE_DNS_OBJ(expect);
	}

	for (size_t i = 0; i < countof(rrs); i++) {
		MDNS_DISPOSE_DNS_OBJ(rrs[i]);
	}
}

@end
